<?php

namespace App\Http\Controllers;

use App\Actions\OrderManagement\CalculateTotalAmountAction;
use App\Http\Requests\OrderManagement\DatatableRequest;
use App\Http\Requests\OrderManagement\OrderStoreRequest;
use App\Http\Requests\OrderManagement\OrderUpdateRequest;
use App\Http\Resources\ProductTypeAheadResource;
use App\Models\Customers;
use App\Models\Employees;
use App\Models\Shop;
use App\Models\Coupon;

use App\Models\CustomerOrder;
use App\Models\OrderManagement;
use App\Models\OrderManagementDetail;
use App\Models\ServiceDetail;
use App\Models\ProductMainStock;

use App\Models\Permission;
use App\Models\Product;
use App\Models\Payment;
use App\Models\ReceivePayments;
use App\Models\ProductPrice;
use App\Models\Category;
use App\Models\PaymentMethod;
use App\Models\BanksOrMobiles;
use App\Models\Account;
use App\Models\InitialTransaction;

use App\Models\District;

use App\Models\Shipper;
use App\Models\Shipment;
use App\Models\ShipmentProduct;
use App\Models\TaxRateSetting;
use App\Models\User;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Datatables;
use Jenssegers\Agent\Agent;
use Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Crypt;
use Carbon\Carbon;
use DateTime;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Response;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Str;

/**
 * @TODO do not include, put KSHER SDK in 'App->Custom' directory
 * @TODO then load KSHER with composer autoloader and use like other classes
 * @TODO you have to put namespace in order to use it
 * @TODO remove all unused variables
 */

/**
 * @TODO Lot of strings without translation method, refactor them like __('translation.Your String')
 */

class OrderManageController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\View\View
     */
    public function index($customerType = '0')
    {
        $sellerId = Auth::user()->id;
        $roleName = Auth::user()->role;
        
        $totalProcessingOrders = OrderManagement::where('seller_id', $sellerId)
                                ->where('order_status', OrderManagement::ORDER_STATUS_PROCESSING)
                                ->customerAsset($roleName)
                                ->count();

        $totalProcessingDropshipperOrders = OrderManagement::where('seller_id', $sellerId)
                                ->where('order_status', OrderManagement::ORDER_STATUS_PROCESSING)
                                ->count();
      
        $statusSchema = OrderManagement::getStatusSchemaForDatatable($roleName, $customerType);

        $defaultStatusOrderId = array_column($statusSchema[0]['sub_status'], 'id')[0];

        $data = [
            'totalProcessingOrders' => $totalProcessingOrders,
            'totalProcessingDropshipperOrders' => $totalProcessingDropshipperOrders,
            'totalProcessingWooCommerce' => '',
            'statusSchema' => $statusSchema,
            'defaultStatusOrderId' => $defaultStatusOrderId,
            'customerType' => $customerType
        ];
        
        return view('seller.order_management.index', $data);
    }

    /**
     * Handle server-side datatable of order managements
     *
     * @param  \App\Http\Requests\OrderManagement\DatatableRequest  $request
     * @return Response
     */
    public function data(DatatableRequest $request)
    {
        $sellerId = Auth::user()->id;

        $orderManagementsTable = (new OrderManagement())->getTable();
        $customersTable = (new Customers())->getTable();

        $orderStatusId = $request->get('status', 0);
       
        //return $orderStatusId;

        $start = $request->get('start', 0);
        $limit = $request->get('length', 100);
        if ($limit < 1 OR $limit > 100) {
            $limit = 100;
        }


        $search = isset($request->get('search')['value'])
                ? $request->get('search')['value']
                : null;

        $orderColumnIndex = isset($request->get('order')[0]['column'])
                            ? $request->get('order')[0]['column']
                            : 1;
        $orderDir = isset($request->get('order')[0]['dir'])
                    ? $request->get('order')[0]['dir']
                    : 'desc';

        $availableColumnsOrder = [
            'id', 'id'
        ];

        $orderColumnName = isset($availableColumnsOrder[$orderColumnIndex])
                            ? $availableColumnsOrder[$orderColumnIndex]
                            : $availableColumnsOrder[0];

        $roleName = Auth::user()->role;

        if(!empty($request->customerType)){
            $customerType = $request->customerType;
        }
        else{
            $customerType = '0';
        }
        
        $orderManagements = OrderManagement::selectRaw("{$orderManagementsTable}.*,
                                {$customersTable}.customer_name AS customer_name,
                                {$customersTable}.company AS company_name,
                                {$customersTable}.phone AS customer_contact_phone")
                //->with('shop')
                //->totalPaid()
                //->withCount('order_management_details')
                ->joinedDatatable()   
                ->searchDatatable($search)
                ->orderBy($orderColumnName, $orderDir)
                ->groupByOrderStatus($orderStatusId)
                ->take($limit)
                ->skip($start)
                ->get();

        $totalData = OrderManagement::searchDatatable($search)->count();

        return Datatables::of($orderManagements)
                ->addIndexColumn()
                ->addColumn('checkbox', function ($row) {
                    return $row->id;
                })
                ->addColumn('order_data', function ($row) use ($orderStatusId) {                    
                    $paymentStatus = '
                        <span class="badge-status--red">
                            '. $row->str_payment_status .'
                        </span>
                    ';

                    $total_discount = !empty($row->total_discount) ? $row->total_discount : 0;

                    $manualPaymentSum = OrderManagement::getManualPaymentSum($row->id);
                    $manualRefundedSum = OrderManagement::getmanualRefundedSum($row->id);
                    $paymentDetailsAllManual = ReceivePayments::with('payment_method')->with('bank_or_mobile_wallet')->where('order_id',$row->id)->get();
  

                    if($row->payment_status == 1)
                        $paymentStatus = '<span class="badge-status--green">PAID</span>';
                    else

                        if($manualPaymentSum == 0 AND $row->payment_status == 0 AND empty($paymentDetailsOthers))
                            $paymentStatus = '<span class="badge-status--red">NOT PAID</span>';
                        

                        if($manualPaymentSum > 0 AND $manualPaymentSum < $row->in_total)
                            $paymentStatus = '<span class="badge-status--yellow">PARTIAL PAID</span>';
                        

                        if($manualPaymentSum == $row->in_total AND $row->order_status == 1)
                            $paymentStatus = '<span class="badge-status--green">PAID</span>';
                        
                        if($row->in_total == $manualPaymentSum AND $row->in_total == $manualRefundedSum)
                            $paymentStatus = '<span class="badge-status--green">NOT PAID</span>';




                    $serviceMethod = '';
                    

                    
                        $confirmBtn = '';
                        $cancelBtn = '';
                        $printLabelBtn = '';
                        $publicPageBtn = '';
                        $shipmentStatusBtns = '';
                        $updateStatusBtnForProcessedAction = '';
                        $updateStatusBtnForProcessed = '<button type="button" class="btn-action--yellow mr-1" title="Update Status"
                                                data-order-id="' . $row->id . '"
                                                data-shipment-id="' . $row->shipment_row_id . '"
                                                onClick="updateStatusForProcessed(this)">
                                                <i class="fa fa-edit mr-2" aria-hidden="true"></i>
                                                Update status
                                            </button>
                                            ';

                                            
                        $array_order_status_id = array('2','8','9');

                        
                    $orderManagementId = $row->id;
                    return '
                        <div class="border border-solid border-gray-400 lg:border-gray-300 rounded-md hover:bg-blue-50">
                            <div class="border border-dashed border-t-0 border-r-0 border-l-0 border-gray-300">
                                <div class="grid grid-cols-3">
                                    <div class="col-span-3 sm:col-span-1">
                                        <a href="'. route('order_management.edit', [ 'order_management' => $row->id ]) .'" data-id="'.$row->id.'" order-status-id="'.$row->order_status.'" class="cursor-pointer underline" title="Edit">
                                            <div class="text-center px-2 py-1 sm:py-2">
                                                <span class="font-bold text-gray-400">#</span>
                                                <span class="relative -left-1 text-blue-500 font-bold">
                                                    '. $row->id .'
                                                </span>
                                            </div>
                                        </a>
                                    </div>
                                   
                                    <div class="col-span-3 sm:col-span-1">
                                        <div class="px-2 py-1 sm:py-2">
                                           

                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div>
                                <div class="grid grid-cols-1 lg:grid-cols-3">
                                    <div class="lg:col-span-2">
                                        <div class="flex justify-between space-x-1">                                            
                                            <div class="col-span-5 sm:col-span-3">
                                                <div class="text-left px-2 py-2">
                                                    <div>
                                                        <span class="text-gray-600">
                                                            Cust. Name :
                                                        </span>'. $row->company_name .'   ['. $row->customer_name .']
                                                    </div>
                                                    <div>
                                                        <span class="text-gray-600">
                                                            
                                                        </span> '. currency_symbol('BDT') . number_format((float)$row->in_total, 2).'
                                                       ( <a data-order-id="' . $row->id . '" data-shipment-id="' . $row->shipment_row_id . '" class="modal-open cursor-pointer" onClick="productsOrdered(this)">' . $row->order_management_details_count .' Item/s</a> )
                                                    </div>
                                                    <div class="mb-2">
                                                        <a data-id="'. $row->id .'" id="BtnAddress" class="modal-open cursor-pointer">' . $serviceMethod .'</a>
                                                    </div>
                                                    
                                                    
                                                    <div class="text-center sm:text-left">
                                                      
                                                    </div>
                                                    <div class="text-center sm:text-left">
                                                       
                                                    </div>
                                                </div>
                                            </div>

                                            <div class="col-span-5 sm:col-span-2 px-2 py-2">
                                                <div class="">
                                                    <span class="text-gray-600 font-bold">
                                                            Discount  :
                                                    </span>'. number_format($total_discount,2) .'
                                                </div>

                                                <div class="">
                                                    <span class="text-gray-600 font-bold">
                                                            Total Paid :
                                                    </span>'. number_format($row->total_paid,2) .'
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="lg:col-span-1">
                                        <div class="border border-dashed border-r-0 border-b-0 border-l-0 border-gray-300">
                                            <div class="flex justify-between space-x-1 px-2 py-1">
                                                <div class="text-xs sm:text-sm">
                                                    <span class="text-gray-600">Order Date :</span> '. date('d/m/Y', strtotime($row->order_date)) .'
                                                </div>
                                                <div class="text-xs sm:text-sm">
                                                    <span class="text-gray-600">Created Date :</span> '. date('d/m/Y h:i a', strtotime($row->created_at)) .'
                                                </div>
                                                <div class="text-xs sm:text-sm">
                                                    <span class="text-gray-600">Created By :</span> '. $row->creator->name .'
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    ';
                })
                ->rawColumns(['checkbox', 'order_data'])
                ->skipPaging()
                ->setTotalRecords($totalData)
                ->make(true);
    }

    /**
     * Show the form for creating a new resource.
     * @return Response
     */
    public function createOrderId($len=16)
    {
        $nonce_str = Str::random($len);
        return $nonce_str;
    }

    /**
     * Show create order_managemenst page
     *
     * @return \Illuminate\View\View
     */
    public function create($customerType = '0')
    {
        $sellerId = Auth::user()->id;

        Session::put('itemArray', []);

        $order_id = '';
        do {
            $order_id = $this->createOrderId();
            $orderManagement = OrderManagement::where('order_id', $order_id)
                                    ->first();
        } while(!empty($orderManagement));

        $products = Product::get();

      
       
        $data = [
            'order_id' => $order_id,
            'customers' => Customers::when(Auth::user()->role=='staff', function ($query){
                            $query->where('shop_id', Auth::user()->shop_id);
                        })->get(),
            'employees' => Employees::when(Auth::user()->role=='staff', function ($query){
                            $query->where('shop_id', Auth::user()->shop_id);
                        })->get(),
            'products' => ProductTypeAheadResource::collection($products),
            'shops' => Shop::all(),
            'customerType' => $customerType,
            'payment_methods' => PaymentMethod::all(),
            'banks_or_mobiles' => BanksOrMobiles::all()
        ];

        return view('seller.order_management.create', $data);
    }



    /* Load Customers Due
    *
    * @return \Illuminate\View\View
    */
   public function customerDue(Request $request)
   {
       $customer_id = $request->customer_id;

       Session::put('itemArray', []);
       $total_due = 0;
       $order_id = '';
       $receivable = InitialTransaction::where('transaction_type','receivable')->where('customer_id',$customer_id)->get();
       if($receivable){
        foreach($receivable as $row){
            $total_due += $row->amount;
        }
       }
       
       $sales = OrderManagement::where('customer_id',$customer_id)->get();
       if($sales){
        foreach($sales as $row){
            $total_due += !empty($row->in_total) ? $row->in_total : 0;
        }
       }

       $payable = InitialTransaction::where('transaction_type','payable')->where('customer_id',$customer_id)->get();
       if($payable){
        foreach($payable as $row){
            $total_due += $row->amount;
        }
       }

       $paid = ReceivePayments::where('customer_id',$customer_id)->get();
       if($paid){
        foreach($paid as $row){
            $total_due -= $row->paid;
        }
       }
       return SettingController::takaBDFormat($total_due);
   }

   public function getCustomerDueById($customer_id)
   {
       Session::put('itemArray', []);
       $total_due = 0;
       $order_id = '';
       $receivable = InitialTransaction::where('transaction_type','receivable')->where('customer_id',$customer_id)->get();
       if($receivable){
        foreach($receivable as $row){
            $total_due += $row->amount;
        }
       }
       
       $sales = OrderManagement::where('customer_id',$customer_id)->get();
       if($sales){
        foreach($sales as $row){
            $total_due += !empty($row->in_total) ? $row->in_total : 0;
        }
       }

       $payable = InitialTransaction::where('transaction_type','payable')->where('customer_id',$customer_id)->get();
       if($payable){
        foreach($payable as $row){
            $total_due += $row->amount;
        }
       }

       $paid = ReceivePayments::where('customer_id',$customer_id)->get();
       if($paid){
        foreach($paid as $row){
            $total_due -= $row->paid;
        }
       }
       return $total_due;
   }

    public function loadCustomers(Request $request){
       
        $data = [
            'customers' => Customers::where('customer_type',$request->customer_type)
                            ->where('shop_id',Auth::user()->shop_id)->get(),            
            'customer_type' => $request->customer_type
        ];
        return view('elements.customer-drodown-or-new', $data);
    }
    

    /**
     * Show create order_managemenst page
     *
     * @return \Illuminate\View\View
     */
    public function loadPaymentTypeData(Request $request)
    {
        $sellerId = Auth::user()->id;       
        $paymentType = $request->paymentType;
        $data = [
            'paymentType' => $paymentType
        ];

        return view('elements.load-payment-type', $data);
    }


    public function moneyReceipt($orderId = '0',$rcvPaymentID)
    {
        $orderManagement = OrderManagement::where('order_id', $orderId)
                            ->with('shop')
                            ->with('customer')
                            ->with(['order_management_details' => function($detail) {
                                $detail->with('product');
                            }])
                            ->with('services_details')
                            ->first();
                        
        $paymentDeatails = ReceivePayments::with('payment_method')->with('bank_or_mobile_wallet')->where('id', $rcvPaymentID)->first();
   
        abort_if(!$orderManagement, Response::HTTP_NOT_FOUND, 'Order not found.');

        $sellerId = $orderManagement->seller_id;
        
        QrCode::generate($orderManagement->order_id, public_path('qrcodes/' . $orderManagement->order_id.'.svg'));
        $manualPaymentSum = OrderManagement::getManualPaymentSum($orderManagement->id);
        $manualRefundedSum = OrderManagement::getmanualRefundedSum($orderManagement->id);
        $paymentDetailsAllManual = ReceivePayments::with('payment_method')->with('bank_or_mobile_wallet')->where('order_id',$orderManagement->id)->get();

        
        $data = [
            'orderManagement' => $orderManagement,            
            'manualPaymentSum' => $manualPaymentSum,            
            'manualRefundedSum' => $manualRefundedSum,            
            'paymentDetailsAllManual' => $paymentDetailsAllManual,            
            'shop_details' => Shop::where('id',Auth::user()->shop_id)->first(),
            'payment_methodBankTransfer' => OrderManagement::PAYMENT_METHOD_BANK_TRANSFER,
            'payment_methodInstant' => OrderManagement::PAYMENT_METHOD_INSTANT,
            'orderStatusPending' => OrderManagement::ORDER_STATUS_PENDING,
            'orderStatusPendingPayment' => OrderManagement::ORDER_STATUS_PENDING_PAYMENT,
            'orderStatusProcessing' => OrderManagement::ORDER_STATUS_PROCESSING,
            'paymentStatusUnPaid' => OrderManagement::PAYMENT_STATUS_UNPAID,
            'paymentStatusPaid' => OrderManagement::PAYMENT_STATUS_PAID,
            'orderStatusPaymentUnconfirmed' => OrderManagement::ORDER_STATUS_PAYMENT_UNCONFIRMED,
            'orderStatusCancel' => OrderManagement::ORDER_STATUS_CANCEL,
            'statusForInfoAlert' => OrderManagement::getStatusForInfoAlert(),
            'paymentDeatails' => $paymentDeatails,
            'taxEnableValues' => OrderManagement::getAllTaxEnableValues(),
            'taxEnableYes' => OrderManagement::TAX_ENABLE_YES,
            'taxRateSetting' => TaxRateSetting::where('seller_id', $sellerId)->first()
        ];
        
        return view('seller.order_management.money-receipt', $data);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param StoreRequest $request
     * @param  \App\Actions\OrderManagement\CalculateTotalAmountAction $calculateTotalAmount
     * @return Response
     */
    public function store(OrderStoreRequest $request, CalculateTotalAmountAction $calculateTotalAmount)
    {
        
        try {
            $sellerId = Auth::user()->id;
            $createdBy = Auth::user()->id;

            $orderId = $request->order_id;

            $taxRateSettingTable = (new TaxRateSetting())->getTable();

            //DB::beginTransaction();
            
            //dd($request->all());

            if(empty($request->customer_id)) {
                $customerId = $this->createCustomer($request,Auth::user()->id,Auth::user()->shop_id);
            }
            
           
            $customer_details = '';
            

        
            $order_date = $request->order_date != null ? date('Y-m-d', strtotime($request->order_date)) : '';
            $orderManagementData = new OrderManagement();
            $orderManagementData->order_date = $order_date;
            $orderManagementData->customer_id = $request->customer_id;
            $orderManagementData->have_services = $request->have_services ?? 0; 
                    
            
            $orderManagementData->order_id = $orderId;
            $orderManagementData->order_status = OrderManagement::ORDER_STATUS_PROCESSING;
               
            $orderManagementData->seller_id= $sellerId;
            //$orderManagementData->encrepted_order_id= Crypt::encryptString($orderId);
            $orderManagementData->created_by= $createdBy;
            $orderManagementData->seller_id= Auth::id();
           
            $orderManagementData->save();
            
            $orderManagementId = $orderManagementData->id;
            
            $paymentUrl = '';           
            $rcvPaymentId = $this->createPayment($request->paid,$orderManagementId,$request);
            

            $productPriceTotal = 0;
            $serviceCostTotal = 0;
            $discountTotal = 0;
            $totalAmount = 0;
            $taxRate = 0;

           
            
            if($request->product_id){
                foreach($request->product_id as $idx => $productId) {
                    $productCostPrice = $request->cost_price[$idx] ?? 0;
                    $productOriginPrice = $request->product_price[$idx] ?? 0;
                    $orderQty = $request->product_qty[$idx] ?? 0;
                    $product_discount = $request->product_discount[$idx] ?? 0;
                    

                    $productPriceTotal += ($productOriginPrice * $orderQty) - $product_discount;


                    $orderManagementDetailData = new orderManagementDetail();
                    $orderManagementDetailData->order_management_id = $orderManagementId;
                    $orderManagementDetailData->product_id = $productId;
                    $orderManagementDetailData->quantity = $orderQty;
                    $orderManagementDetailData->price = $productOriginPrice;
                    $orderManagementDetailData->product_discount = $product_discount;
                    $orderManagementDetailData->seller_id = $sellerId;
                    $orderManagementDetailData->created_at = new DateTime();
                    
                    $orderManagementDetailData->save();
                    
                    
                    // if($orderManagementDetailData){
                    //     $productMainStock = ProductMainStock::where('product_id',$productId)->first();
                    
                    //     $productMainStock->quantity = $productMainStock->quantity-$orderQty;
                    //     $productMainStock->save();
                    // }

                }
            }

    

            if($request->have_services){
                if($request->service_name){
                    foreach ($request->service_name as $idx => $serviceName) {
                        $servicePrice = $request->service_price[$idx] ?? 0;
                        $ram_id = $request->ram_id[$idx] ?? 0;
                        $employee_id = $request->employee_id[$idx] ?? 0;
                    
                        $serviceCostTotal += $servicePrice;

                        $serviceDetailData = new ServiceDetail();
                        $serviceDetailData->order_management_id = $orderManagementId;
                        $serviceDetailData->service_name = $serviceName;
                        $serviceDetailData->shop_id = Auth::user()->shop_id;
                        $serviceDetailData->price = $servicePrice;
                        $serviceDetailData->ram_id = $ram_id;
                        $serviceDetailData->employee_id = $employee_id;
                        $serviceDetailData->seller_id = $sellerId;
                        $serviceDetailData->save();
                    }
                }
            }


            $orderManagementData = OrderManagement::where('id', $orderManagementId)->first();   

            $orderManagementDtl = OrderManagement::select('in_total', 'payment_status')->where('id',$orderManagementId)->first();
            $manualPaymentSumDtl = OrderManagement::getManualPaymentSum($orderManagementId);
            $manualRefundedSumDtl = OrderManagement::getManualRefundedSum($orderManagementId);
            $totalPaidAmount = $manualPaymentSumDtl - $manualRefundedSumDtl;
            

            $totalAmount = $calculateTotalAmount->handle($productPriceTotal, $taxRate, $serviceCostTotal, $request->total_discount);
            
            
           
            $orderManagementData->product_price_total = $productPriceTotal;
            $orderManagementData->service_price_total = $serviceCostTotal;
            $orderManagementData->in_total = $totalAmount; 
            $orderManagementData->total_discount = $request->total_discount;           

            
            if($orderManagementDtl->in_total == $totalPaidAmount){
               
                $orderManagementData->payment_status = OrderManagement::PAYMENT_STATUS_PAID;
                $orderManagementData->order_status = OrderManagement::ORDER_STATUS_COMPLETED;
                
            }

            if($orderManagementDtl->in_total == $manualRefundedSumDtl){
                    $orderManagementData->payment_status = OrderManagement::PAYMENT_STATUS_UNPAID;
                    $orderManagementData->order_status = OrderManagement::ORDER_STATUS_PENDING;
            }

            $orderManagementData->save();
            
            //DB::commit();

            $responseData = [
                'orderId' => $orderManagementId,
                // 'productPriceTotal' => $productPriceTotal,
                // 'serviceCost' => $serviceCostTotal,
                // 'discountTotal' => $discountTotal,
                // 'totalAmount' => $totalAmount,
                // 'totalPaid' => $request->paid,  
                // 'totalDue' => $totalAmount-$request->paid,               
                // 'publicUrl' => route('order-management.public-url', [ 'order_id' => $orderId ]),
                //'moneyReceipt' => route('order_management.money_receipt', [ 'order_id' => $orderId ,'receive_payment_id' => $rcvPaymentId ]),
                
            ];
            if($orderManagementId){
                return response()->json([
                    'status' => 1,
                    'message' => '<span class="flex space-x-4 text-green-600">Order created successfully</span>'
                ]);
            }
            

            //return $this->apiResponse(Response::HTTP_CREATED, 'Order created.', $responseData);

        } catch (\Throwable $th) {
            report($th);
            DB::rollBack();

            return redirect()->back()
                    ->with('error', $th->getMessage())
                    ->withInput();
        }
        /**
         * @TODO instead of Throwable, use Exception
         */
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return Response
     */
    public function edit($id)
    {
        $sellerId = Auth::user()->id;
        
        $orderManagement = OrderManagement::where('id', $id)
                            ->with('shop')
                            //->with('coupon_details')
                            ->with('services_details')
                            ->with(['order_management_details' => function($detail) {
                                $detail->with('product');
                            }])
                            ->first();
                          
        abort_if(!$orderManagement, 404, __('translation.Data not found'));

        if ($orderManagement->customer->customer_type == 'show_room')
            $customers = DB::connection('mysql_secondary')->select('SELECT * FROM customers');
        else
            $customers = Customers::get();     

        $products = Product::get();

        $enabledServiceMethodIds = [];
        


        


        $addedProductCodes = $orderManagement->order_management_details->map(function($detail) {
            return $detail->product->id;
        });

        $manualPaymentSum = OrderManagement::getManualPaymentSum($id);

        $manualRefundedSum = OrderManagement::getmanualRefundedSum($id);
        $paymentDetailsAllManual = ReceivePayments::with('payment_method')
        ->with(['bank_or_mobile_wallet' => function($detail) {
            $detail->with('accounts');
        }])
        ->with('bank_or_mobile_wallet_account')
        ->where('order_id',$id)->get();
        
        $paymentDetailsOthers = ReceivePayments::where('order_id', $id)->first();

        $getOrderStatus = OrderManagement::getOrderStatus($orderManagement->order_status);

        //dd($paymentDetailsAllManual);
        $data = [
            'order_id' => $id,
            'manualPaymentSum' => $manualPaymentSum,
            'manualRefundedSum' => $manualRefundedSum,
            'paymentDetailsAllManual' => $paymentDetailsAllManual,
            'paymentDetailsOthers' => $paymentDetailsOthers,
            'getOrderStatus' => $getOrderStatus,
            'orderStatuses' => OrderManagement::getAllAvailableStatusForEdit(),
            'paymentStatuses' => OrderManagement::getAllPaymentStatus(),
            'employees' => Employees::where('shop_id',Auth::user()->shop_id)->get(),
            'products' => ProductTypeAheadResource::collection($products),
            'orderManagement' => $orderManagement,
            'addedProductCodes' => $addedProductCodes,
            'shops' => Shop::all(),           
            'customers' => $customers,
            'payment_methods' => PaymentMethod::all(),
            'banks_or_mobiles' => BanksOrMobiles::all(),
            'accounts' => Account::get(),
            'customerDue' => $this->getCustomerDueById($orderManagement->customer_id),
        ];

        
        return view('seller.order_management.edit', $data);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \App\Http\Requests\OrderManagement\UpdateRequest  $request
     * @param  \App\Actions\OrderManagement\CalculateTotalAmountAction  $calculateTotalAmount
     * @return Response
     */
    public function update(OrderUpdateRequest $request, CalculateTotalAmountAction $calculateTotalAmount)
    {
        
        try {
            $sellerId = Auth::user()->id;
           
            $orderManagementId = $request->id;
            $orderManagementData = OrderManagement::where('id', $orderManagementId)->first();

            $orderManagementDetailsTable = (new OrderManagementDetail())->getTable();

            DB::beginTransaction();

            if (empty($request->customer_id)) {
                $customerId = $this->createCustomer($request,Auth::user()->id,Auth::user()->shop_id);
            }else{
                $customerId = $request->customer_id;
            }

            $taxRate = 0;

            $productPriceTotal = 0;
            $serviceCostTotal = 0;
            $discountTotal = 0;
            $totalAmount = 0;

            
            DB::table($orderManagementDetailsTable)
                ->where('order_management_id', $orderManagementId)
                ->delete();

                
            if($request->product_id){
                foreach ($request->product_id as $idx => $productId) {
                    $productCostPrice = $request->cost_price[$idx] ?? 0;
                    $productOriginPrice = $request->product_price[$idx] ?? 0;
                    $orderQty = $request->product_qty[$idx] ?? 0;
                    $product_discount = $request->product_discount[$idx] ?? 0;
                    

                    $productPriceTotal += ($productOriginPrice * $orderQty) - $product_discount;


                    $orderManagementDetailData = new orderManagementDetail();
                    $orderManagementDetailData->order_management_id = $orderManagementId;
                    $orderManagementDetailData->product_id = $productId;
                    $orderManagementDetailData->shop_id = Auth::user()->shop_id;
                    $orderManagementDetailData->quantity = $orderQty;
                    $orderManagementDetailData->price = $productOriginPrice;
                    $orderManagementDetailData->product_discount = $product_discount;
                    $orderManagementDetailData->seller_id = $sellerId;
                    $orderManagementDetailData->created_at = new DateTime();
                    
                    $orderManagementDetailData->save();
                    
                    // if($orderManagementDetailData){
                    //     $productMainStock = ProductMainStock::where('product_id',$productId)->first();
                    
                    //     $productMainStock->quantity = $productMainStock->quantity-$orderQty;
                    //     $productMainStock->save();
                    // }

                }
            }

           
         


                $serviceDetailTable = (new ServiceDetail())->getTable();
                DB::table($serviceDetailTable)
                    ->where('order_management_id', $orderManagementId)
                    ->delete();
                
                if($request->have_services){
                    if($request->service_name){
                        foreach($request->service_name as $idx => $serviceName) {
                            $servicePrice = $request->service_price[$idx] ?? 0;
                            $serviceCostTotal += $servicePrice;
                

                        $serviceDetailData = new ServiceDetail();
                        $serviceDetailData->order_management_id = $orderManagementId;
                        $serviceDetailData->service_name = $serviceName;
                        $serviceDetailData->price = $servicePrice;                       
                        $serviceDetailData->seller_id = $sellerId;
                        $serviceDetailData->created_at = new DateTime();
                        $serviceDetailData->save();

                    
                    }
                }
            }
            

            
            $totalAmount = $calculateTotalAmount->handle($productPriceTotal, $taxRate, $serviceCostTotal, $request->total_discount);

            $orderManagementData->order_date = $request->order_date != null ? date('Y-m-d', strtotime($request->order_date)) : date('Y-m-d');
            $orderManagementData->customer_id = $customerId;
            $orderManagementData->shop_id = $request->shop_id;  
            $orderManagementData->have_services = $request->have_services ?? 0;            
            $orderManagementData->product_price_total = $productPriceTotal;               
            $orderManagementData->service_price_total = $serviceCostTotal;               
            $orderManagementData->total_discount = $request->total_discount;               
            $orderManagementData->in_total = $totalAmount;               
            $orderManagementData->seller_id= $sellerId;
            $orderManagementData->created_by= Auth::id();
            
            $orderManagementData->updated_at = new DateTime();            
            $orderManagementData->save();


            $rcvPaymentId = $this->createPayment($request->paid,$orderManagementId,$request);

            DB::commit();

            $responseData = [
                'productPriceTotal' => $productPriceTotal,
                'serviceCost' => $serviceCostTotal,
                'discountTotal' => $discountTotal,
                'totalAmount' => $totalAmount
            ];

            if($orderManagementId){
                return response()->json([
                    'status' => 1,
                    'message' => '<span class="flex space-x-4 text-green-600">Order Updated successfully</span>'
                ]);
            }

        } catch (\Throwable $th) {
            report($th);

            DB::rollBack();

            return redirect()->back()
            ->with('error', $th->getMessage())
            ->withInput();
        }
    }

    public function orderManagementUpdate(Request $request, $id)
    {
        $orderManagement = OrderManagement::find($id);

        $orderManagement->shop_id = $request->shop_id;

        $orderManagement->customer_id = $customer->id;
        $orderManagement->order_status = $request->order_status;
        $orderManagement->payment_status = $request->payment_status;
        $orderManagement->total_discount = $request->total_discount;
        $orderManagement->total_product_weight = $request->total_product_weight;
        $orderManagement->created_by = Auth::user()->id;
        $orderManagement->save();

        OrderManagementDetail::where('order_management_id',$orderManagement->id)->delete();

        if(count($request->product_id)>0)
        {
            foreach($request->product_id as $key=>$row)
            {
                $orderManagementDetails = new OrderManagementDetail();
                $orderManagementDetails->product_id = $row;
                $orderManagementDetails->quantity = $request->product_quantity[$key];
                $orderManagementDetails->shop_id = Auth::user()->shop_id;
                $orderManagementDetails->product_discount = $request->product_discount[$key];
                $orderManagementDetails->seller_id = Auth::user()->id;
                $orderManagementDetails->order_management_id = $orderManagement->id;
                $orderManagementDetails->save();
            }
        }

        if($orderManagement)
        {
            return redirect('order_management')->with('success','Your Order Updated Successfully');
        }
        else{
            return redirect('order_management')->with('danger','Something happened wrong');
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return Response
     */
    public function destroy($id)
    {
        //
    }

    public function orderManagementDelete($id)
    {
        $orderPurchase = OrderManagement::where('id',$id)->where('shop_id',Auth::user()->shop_id)->delete();
        OrderManagementDetail::where('order_management_id',$id)->where('shop_id',Auth::user()->shop_id)->delete();
        if($orderPurchase)
        {
            return redirect()->back()->with('success','Order Purchase Deleted Successfully');
        }
        else{
            return redirect()->back()->with('danger','Something happened wrong');
        }
    }

    public function orderManagementCancel(Request $request)
    {
        try {
            $orderManagement = OrderManagement::where('id', $request->id)->first();
            $orderManagement->order_status = OrderManagement::ORDER_STATUS_CANCEL;
            $orderManagement->save();

            $orderManagementDetail = OrderManagementDetail::where('order_management_id', $request->id)->get();

            if($orderManagementDetail){
                foreach($orderManagementDetail as $item){
                    $productMainStock = ProductMainStock::where('product_id', $item->product_id)->first();
                    $productMainStock->quantity = $productMainStock->quantity+$item->quantity;
                    $productMainStock->save();                    
                }
            }

            ReceivePayments::where('order_id', $request->id)->delete();

            

           


            return [
                'status' => 1
            ];

        } catch (\Throwable $th) {
            report($th);

            return $this->apiResponse(Response::HTTP_INTERNAL_SERVER_ERROR, 'Something went wrong');
        }
    }

    public function getOrderedProducts(Request $request)
    {
        $orderId = $request->get('orderId', 0);
        $shipmentId = $request->get('shipmentId', 0);

        $orderDetails = OrderManagementDetail::where('order_management_id', $orderId)->where('shop_id',Auth::user()->shop_id)->with('product')->get();
        $shipments = ShipmentProduct::where('shipment_id', $shipmentId)->get();

        $productData = [];

        foreach ($orderDetails as $item) {

            $row = [];
            $row[] = '
                <div class="grid grid-cols-1 gap-4">
                    <div>
                        <img src="'. $item->product->image_url .'" class="w-16 md:w-11/12 h-auto" />
                    </div>
                    <div>
                        <span class="whitespace-nowrap text-blue-500">
                            ID : <strong>'. $item->product->id .'</strong>
                        </span>
                    </div>
                </div>
            ';

            $productPrice = $item->price - $item->discount_price;
            $shippedProducts = $item->quantity;
            foreach ($shipments as $shipment){
                if ($shipment->product_id == $item->product_id){
                    $shippedProducts = $shipment->quantity;
                }
            }

            $shippedProductsContents = '';
            if (!empty($shipmentId)){
                $shippedProductsContents = '
                    <div class="mb-1">
                        <div class="whitespace-nowrap">
                            <label class="text-gray-700">
                                Shipment Qty :
                            </label>
                            <span class="text-gray-900">
                                '. number_format($shippedProducts) .'
                            </span>
                        </div>
                    </div>
                    ';
            }

            $row[] = '
                <div>
                    <div class="mb-1">
                        <strong class="text-blue-500">'. $item->product->product_code .'</strong>
                    </div>
                    <div class="mb-1">
                        <div class="whitespace-nowrap">
                            <label class="text-gray-700">
                                Price :
                            </label>
                            <span>'. currency_symbol('BDT') . number_format(floatval($productPrice), 2) .'</span>
                        </div>
                    </div>
                    <div class="mb-1">
                        <div class="whitespace-nowrap">
                            <label class="text-gray-700">
                                Ordered Qty :
                            </label>
                            <span class="text-gray-900">
                                '. number_format($item->quantity) .'
                            </span>
                        </div>
                    </div>
                   ' . $shippedProductsContents . '
                    <div class="mb-1">
                        <div class="whitespace-nowrap">
                            <label class="text-gray-700">
                                Total Price :
                            </label>
                            <strong class="">'. currency_symbol('BDT') . number_format(floatval($productPrice * $shippedProducts), 2) .'</strong>
                        </div>
                    </div>
                </div>
            ';

            $productData[] = $row;
        }

        return response()->json([
            'data' => $productData,
        ]);
    }

    public function getServicesItems(Request $request)
    {
        $orderId = $request->get('orderId', 0);

        $orderDetails = ServiceDetail::where('order_management_id', $orderId)->where('shop_id',Auth::user()->shop_id)->with('product')->get();

        $productData = [];

        foreach ($orderDetails as $item) {

            $row = [];
            $row[] = '
                <div class="grid grid-cols-1 gap-4">
                    <div>
                        <span class="whitespace-nowrap text-blue-500">
                            ID : <strong>'. $item->id .'</strong>
                        </span>
                    </div>
                </div>
            ';

            $productPrice = $item->price - $item->discount_price;

            $row[] = '
                <div>
                    <div class="mb-1">
                        <strong>'. $item->service_name .'</strong>
                    </div>
                    <div class="mb-1">
                        <div class="whitespace-nowrap">
                            <label class="text-gray-700">
                                Price :
                            </label>
                            <span>'. currency_symbol('BDT') . number_format(floatval($item->price), 2) .'</span>
                        </div>
                    </div>
                    <div class="mb-1">
                        <div class="whitespace-nowrap">
                            <label class="text-gray-700">
                                Discount :
                            </label>
                            <span class="text-gray-900">
                                '. number_format($item->discount_price) .'
                            </span>
                        </div>
                    </div>
                    <div class="mb-1">
                        <div class="whitespace-nowrap">
                            <label class="text-gray-700">
                                Total Price :
                            </label>
                            <strong class="">'. currency_symbol('BDT') . number_format(floatval($productPrice), 2) .'</strong>
                        </div>
                    </div>
                </div>
            ';

            $productData[] = $row;
        }

        return response()->json([
            'data' => $productData,
        ]);
    }

    /**
     * @TODO put private key in a file and read it here
     */
    public function private_key(){
        $privatekey=<<<EOD
    -----BEGIN RSA PRIVATE KEY-----
    MIICYgIBAAKBgQCPzwGZv5sCMwf8Sv+FXUqrULSEdeB846z2OCnPw+ynDTUqApRz
    0Goj1gYaK5Gu4vLxTH06PpL96sAB9C0pACBz3xewotdAwoHK0B86TaWk0bt4+jSL
    HMAvgLOF2DH5uAlDzYp8KtQAyhXOowds/20POw+Q3m2RgLCMXQ4OzElp8QIDAQAB
    AoGBAI4VecBdZhp7LwWfV+x9axvuRhyllmHuVOKERRNIwZWfYAqct+3hWi0D9c1/
    hJWlF2E/MG8Oig6kFIcZp5OwAvIHsEkJjryQSk4qERpuU99TG9u5ayGmFUPaC0x6
    fzgEw3+ANYOytWTfsxGbUL1SFoZ1yqKD/iKuBE2BXgM6fZbBAkUAv3jyTVA5R+kg
    B3eFSu+hywi87Q2zZ+myBHGBC4Zb3mhmKRoiBMGZS40y9JXNsmrx3IhynQDSiywJ
    7DyX+Bo7SJ90eykCPQDAReuuYuU/wqcqtnscRzVCW9aydquaDYUHOUXWsAGdghtK
    SJFJW717RLHO/3L230f2pl5TBfPG3hGYmYkCRA8O0e9mmbqgCNbNfXwRMGYpP8Jc
    y3kmlctnqcBgRqVNDIu69GXvW8DnT9SQW2bmpjKzwF+8itJLGlSrxz/JwFPLxntR
    Aj0AjT1PqaSAHtxQjDHMMbOlTf/EsQg3ekzgIbRStyhHp3qBrYmtICRCBqEptJM1
    0l+mr2r68yX2M2nBp0VxAkUAkT6IL2UAbBi5mTK2YgakqyWCcFsLg7fGtArKcNiF
    QssbrooyyUHq8GKQ/4IYQO6M80xTf6vY3r3Gxs8LkqoQirHwRN0=
    -----END RSA PRIVATE KEY-----
    EOD;

    return $privatekey;
    }


    public function createCustomer($request){
        $present_arr = array();
        $present_address = '';
        $present_arr['present_district'] = $request->present_district;
        $present_arr['present_thana'] = $request->present_thana;
        $present_arr['present_postoffice'] = $request->present_post_office;
        $present_arr['present_postcode'] = $request->present_post_code;
        $present_address = json_encode($present_arr);  
        
        $permanent_arr = array();
        $permanent_address = '';
        $same_as_present = 0;
        if (isset($request->same_as_present) && $request->same_as_present === 'on') {
            $same_as_present = 1;
            $permanent_address = '';
        }else{  
            $permanent_arr['permanent_district'] = $request->permanent_district;
            $permanent_arr['permanent_thana'] = $request->permanent_thana;
            $permanent_arr['permanent_postoffice'] = $request->permanent_post_office;
            $permanent_arr['permanent_postcode'] = $request->permanent_post_code; 
            
            $permanent_address = json_encode($permanent_arr); 
        }
    
    

        $customer = Customers::create([
            'customer_type' => $request->customer_type,
            'customer_name' => $request->customer_name,
            'present_address' => $present_address,
            'same_as_present' => $same_as_present,
            'permanent_address' => $permanent_address,
            'phone' => $request->phone,
            'nationality' => 'Bangladeshi',
            'seller_id' => Auth::id(),
            'shop_id' => Auth::user()->shop_id,
        ]);

        return $customer->id;
    }

    /**
     * @TODO put private key in a file and read it here
     */
    public function createPayment($payment_amount,$order_id,$request)
    {
        
        $payment_date = $request->order_date != null ? date('Y-m-d', strtotime($request->order_date)) : date('Y-m-d');
        
        if($request->is_cash_payment){
            $payment_method = PaymentMethod::where('name','Cash')->first();
            $payment_method_id = $payment_method->id;
        }else{
            $payment_method_id = $request->payment_method_id; 
        }

        if($payment_amount>0){
            $paymentDetails = new ReceivePayments();
            $paymentDetails->amount = $payment_amount;
            $paymentDetails->paid = $payment_amount;
            $paymentDetails->is_confirmed = 1;
            $paymentDetails->type = 'customer';
            $paymentDetails->shop_id = Auth::user()->shop_id;
            $paymentDetails->order_id = $order_id;
            $paymentDetails->customer_id = $request->customer_id;
            $paymentDetails->payment_date = $payment_date;
            $paymentDetails->payment_time = Carbon::now()->format('h:i');
            $paymentDetails->payment_method_id = $payment_method_id;
            $paymentDetails->bank_or_mobile_wallet_id = $request->bank_or_mobile_wallet_id;
            $paymentDetails->account_no_id = $request->account_no_id ?? 0;
            $result = $paymentDetails->save();
            return $paymentDetails->id;
        }else{
            return 0;
        }
        
    }


    public function orderStatus($order_id)
    {
         $orderManagement = OrderManagement::where('order_id',$order_id)->first();
         set_time_limit(0);
         $random_str = $this->generateRandomString();

         
        $orderManagement->payment_status = 1;
        $orderManagement->order_status = 2;
        $orderManagement->payment_date  = $gateway_pay_array['data']['time_end'];
        $result = $orderManagement->save();

        return Redirect::to('/sales_management_customer/'.$order_id)->with('success', 'Your Order Updated Successfully');
        
    }

    public function getSubCatName(){
        $subCategoryID = $_GET['subCategoryID'];
        $cats = Category::select('cat_name')->where('id',$subCategoryID)->first();
        if($cats){
            return $result = array(
                'cat_name'=>$cats->cat_name
            );
        }
        else{
            return 0;
        }
    }


    public function bulkStatus(Request $request)
    {
        if ($request->ajax()) {

            if (isset($request->jSonData) && $request->jSonData != null) {
                $arr = json_decode($request->jSonData);
                foreach ($arr as $webIDOrderID) {

                    //SAVE TO LOCAL DB
                    $orderManagement = OrderManagement::find($webIDOrderID);
                    $orderManagement->order_status = $request->status;
                    $orderManagement->save();
                    $flag = 1;
                }

                if($flag==1){
                    echo 'Order Status Changed Successfully';
                }
            }
        }
    }




    public function getOrderHistory(Request $request){
        $editData = OrderManagement::where('id',$request->order_id)->with('orderProductDetails')->where('shop_id',Auth::user()->shop_id)->first();

        $data = [];
        if(isset($editData->orderProductDetails) && count($editData->orderProductDetails)>0){
            $priceAndShop = [];
            foreach($editData->orderProductDetails as $key=>$row)
            {
                $price = ProductPrice::where('shop_id',Auth::user()->shop_id)->where('product_id',$row->product->id)->where('shop_id',$row->shop_id)->first();

                if(!empty($price->price))
                {
                    $priceAndShop[$key]['price'] = $price->price;
                }
                else
                {
                    $priceAndShop[$key]['price'] = '';
                }
                if(!empty($price->shop_id))
                {
                    $priceAndShop[$key]['shop'] = $price->shop_id;
                }
                else
                {
                    $priceAndShop[$key]['shop'] = '';
                }
                if(!empty($row->shop_id))
                {
                    $getShopId = 'shop_id'.$row->shop_id;
                }
                else{
                    $getShopId = '';
                }

                array_push($data,$row->product->part_no.$getShopId);
            }
        }
        Session::put('itemArray',$data);
        return view('seller.order_management.getOrderHistory', compact('editData', 'priceAndShop'));
    }

 

    public function orderDelete(Request $request)
    {
        $OrderManagement = OrderManagement::where('id',$request->id)->delete();
        OrderManagement::where('order_id',$request->id)->where('shop_id',Auth::user()->shop_id)->delete();
        if($OrderManagement)
        {
             return [
                'status' => 1
             ];
        }
     }

    /**
     * @TODO not sure why you're detecting mobile usage, refactor or make helper function for that
     */
    public function isMobile(){
        return preg_match("/(android|avantgo|blackberry|bolt|boost|cricket|docomo|fone|hiptop|mini|mobi|palm|phone|pie|tablet|up\.browser|up\.link|webos|wos)/i", $_SERVER["HTTP_USER_AGENT"]);
    }


    /**
     * @TODO Like I mentioned above, use built-in method for random string
     */
    function generateRandomString($length = 10) {
        return substr(str_shuffle(str_repeat($x='0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ', ceil($length/strlen($x)) )),1,$length);
    }

    public function getOrderPaymentDetails(Request $request){
        $paymentDetails = ReceivePayments::where('order_id',$request->order_id)->first();

        $data = [
            'amount'=>$paymentDetails->amount,
            'paid'=>$paymentDetails->amount,
            'payment_date'=>date('d/m/Y', strtotime($paymentDetails->payment_date)),
            'payment_time'=>$paymentDetails->payment_time,
            'payment_slip'=>$paymentDetails->payment_slip
        ];
        return $data;


    }


    public function makeNewPayment(Request $request){

        
            $payment_date = $request->payment_date != null ? date('Y-m-d', strtotime($request->payment_date)) : '';
            $paymentDetails = new ReceivePayments();
            $paymentDetails->shop_id = Auth::user()->shop_id;
            $paymentDetails->amount = $request->payment_amount;
            $paymentDetails->paid = $request->payment_amount;
            $paymentDetails->is_confirmed = $request->is_confirmed;
            $paymentDetails->order_id = $request->order_id;
            $paymentDetails->payment_date = $payment_date;
            $paymentDetails->payment_time = Carbon::now()->format('h:i');
            $paymentDetails->payment_method_id = $request->payment_method_id;
            $paymentDetails->bank_or_mobile_wallet_id = $request->bank_or_mobile_wallet_id;
            $paymentDetails->account_no_id = $request->account_no_id ?? 0;
            $paymentDetails->is_refund = $request->is_refund;
            $result = $paymentDetails->save();

            if($result){
                $orderManagementDtl = OrderManagement::select('in_total', 'payment_status')->where('id',$request->order_id)->first();
                $manualPaymentSumDtl = OrderManagement::getManualPaymentSum($request->order_id);
                $manualRefundedSumDtl = OrderManagement::getManualRefundedSum($request->order_id);
                $totalPaidAmount = $manualPaymentSumDtl - $manualRefundedSumDtl;

                if($orderManagementDtl->in_total == $totalPaidAmount){
                    $orders = OrderManagement::find($request->order_id);
                     $orders->payment_status = OrderManagement::PAYMENT_STATUS_PAID;
                     $orders->order_status = OrderManagement::ORDER_STATUS_COMPLETED;
                     $orders->save();
                }

                if($orderManagementDtl->in_total == $manualRefundedSumDtl){
                    $orders = OrderManagement::find($request->order_id);
                     $orders->payment_status = OrderManagement::PAYMENT_STATUS_UNPAID;
                     $orders->order_status = OrderManagement::ORDER_STATUS_PENDING;
                     $orders->save();
                }
             }

            $orderManagement = OrderManagement::select('order_id','in_total', 'payment_status')->where('id',$request->order_id)->first();
            $manualPaymentSum = OrderManagement::getManualPaymentSum($request->order_id);
            $manualRefundedSum = OrderManagement::getmanualRefundedSum($request->order_id);

            $paymentDetailsAllManual = ReceivePayments::where('order_id',$request->order_id)->get();

            return view('seller.order_management.payment_table', compact('manualPaymentSum', 'paymentDetailsAllManual', 'orderManagement', 'manualRefundedSum'));
    }

    /**
     * @TODO Go through the statements, remove unused or commented codebase
     */
    public function updateManualPayment(Request $request){


            
            $paymentDetails = ReceivePayments::find($request->payment_id);
            
            $paymentDetails->amount = $request->amount;
            $paymentDetails->paid = $request->amount;
            $paymentDetails->is_confirmed = $request->is_confirmed;
            $paymentDetails->payment_method_id = $request->payment_method_id;
            $paymentDetails->bank_or_mobile_wallet_id = $request->bank_or_mobile_wallet_id;
            $paymentDetails->account_no_id = $request->account_no_id ?? 0;
            $paymentDetails->is_refund = $request->is_refund;

            $result = $paymentDetails->save();

             if($result){
                $orderManagementDtl = OrderManagement::select('in_total', 'payment_status')->where('id',$request->order_id)->first();
                
                $manualPaymentSumDtl = OrderManagement::getManualPaymentSum($request->order_id);
                $manualRefundedSumDtl = OrderManagement::getManualRefundedSum($request->order_id);
                $totalPaidAmount = $manualPaymentSumDtl - $manualRefundedSumDtl;
                
                if($orderManagementDtl->in_total == $totalPaidAmount){
                     $orders = OrderManagement::find($request->order_id);
                     $orders->payment_status = OrderManagement::PAYMENT_STATUS_PAID;
                     //$orders->payment_date  = Carbon::parse($request->payment_date)->format('Y-m-d h:i:s');
                     $orders->order_status = OrderManagement::ORDER_STATUS_COMPLETED;
                     $orders->save();
                }
                
                if($orderManagementDtl->in_total == $manualRefundedSumDtl){
                    $orders = OrderManagement::find($request->order_id);
                     $orders->payment_status = OrderManagement::PAYMENT_STATUS_UNPAID;
                     $orders->order_status = OrderManagement::ORDER_STATUS_PENDING;
                     $orders->save();
                }

            }

            $orderManagement = OrderManagement::select('in_total', 'payment_status')->where('id',$request->order_id)->first();
            $manualPaymentSum = OrderManagement::getManualPaymentSum($request->order_id);
            $manualRefundedSum = OrderManagement::getmanualRefundedSum($request->order_id);
            $paymentDetailsAllManual = ReceivePayments::where('order_id',$request->order_id)->get();

            return view('seller.order_management.payment_table', compact('manualPaymentSum', 'paymentDetailsAllManual', 'orderManagement', 'manualRefundedSum'));
    }









    public function getOrderedProductDetails(Request $request){
        $order_id = $request->orderId;
        $product_id = $request->product_id;
        $editData = Product::where('id',$request->product_id)->where('shop_id',Auth::user()->shop_id)->first();

        $shipped_qty = '';
        $product_price = '';
        if(isset($editData)){

            $getShippedQty = ShipmentProduct::where('order_id', $request->orderId)
            ->where('product_id', $request->product_id)
            ->sum('quantity');

            if(!empty($getShippedQty)){
                $shipped_qty = $getShippedQty;
            }
            else{
                $shipped_qty = 0;
            }

        }

        return view('seller.order_management.orderedProductedDetails', compact('order_id', 'editData', 'shipped_qty', 'product_price'));
    }


}
